/*
 * Copyright (C) 2002 Benjamin Hummel (benjamin@datamaze.de)
 *
 * camera.c - support functions for 1394 camera control
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>

#include <libdc1394/dc1394_control.h>
#include <libraw1394/raw1394.h>

#include "camera.h"
#include "error.h"

raw1394handle_t handle = 0;
nodeid_t *camera_nodes = 0;
int num_cameras = 0;
int current_cam = 0;
nodeid_t current_node = 0;

/* list of all features */
struct lookup_struct feature_table[] =
{
	{ FEATURE_BRIGHTNESS,       "brightness"       },
	{ FEATURE_EXPOSURE,         "exposure"         },
	{ FEATURE_SHARPNESS,        "sharpness"        },
	{ FEATURE_WHITE_BALANCE_UB, "white_balance_ub" },
	{ FEATURE_WHITE_BALANCE_VR, "white_balance_vr" },
	{ FEATURE_HUE,              "hue"              },
	{ FEATURE_SATURATION,       "saturation"       },
	{ FEATURE_GAMMA,            "gamma"            },
	{ FEATURE_SHUTTER,          "shutter"          },
	{ FEATURE_GAIN,             "gain"             },
	{ FEATURE_IRIS,             "iris"             },
	{ FEATURE_FOCUS,            "focus"            },
	{ FEATURE_TEMPERATURE,      "temperature"      },
	{ FEATURE_TRIGGER,          "trigger"          },
	{ FEATURE_ZOOM,             "zoom"             },
	{ FEATURE_PAN,              "pan"              },
	{ FEATURE_TILT,             "tilt"             },
	{ FEATURE_OPTICAL_FILTER,   "optical_filter"   },
	{ FEATURE_CAPTURE_SIZE,     "capture_size"     },
	{ FEATURE_CAPTURE_QUALITY,  "capture_quality"  }
};
int feature_table_size = sizeof (feature_table) / sizeof (struct lookup_struct);

/* macros to simplify the generation of the alias table */
#define GEN_UNDERSCORE(feat,p1,p2) {feat, p1 p2}, {feat, p1 "_" p2}
#define GEN_WHITE_MIX(feat,p1,p2,x,y) \
	GEN_UNDERSCORE(feat,p1,p2 x),   GEN_UNDERSCORE(feat,p1,p2 "_" x), \
	GEN_UNDERSCORE(feat,p1,p2 y),   GEN_UNDERSCORE(feat,p1,p2 "_" y), \
	GEN_UNDERSCORE(feat,p1,p2 x y), GEN_UNDERSCORE(feat,p1,p2 "_" x y), \
	GEN_UNDERSCORE(feat,p1,p2 y x), GEN_UNDERSCORE(feat,p1,p2 "_" y x)
#define GEN_WHITE_BALANCE(feat,x,y) \
	GEN_WHITE_MIX(feat, "white", "balance",x,y), \
	GEN_WHITE_MIX(feat, "white", "bal",x,y), \
	GEN_WHITE_MIX(feat, "w", "b",x,y)

/* the alias table */
struct lookup_struct feature_alias[] =
{
	{ FEATURE_BRIGHTNESS, "bright" },
	{ FEATURE_BRIGHTNESS, "bri" },
	{ FEATURE_EXPOSURE, "exp" },
	{ FEATURE_SHARPNESS, "sharp" },

	GEN_WHITE_BALANCE (FEATURE_WHITE_BALANCE_UB, "u", "b"),
	GEN_WHITE_BALANCE (FEATURE_WHITE_BALANCE_VR, "v", "r"),

	{ FEATURE_SATURATION, "sat" },
	{ FEATURE_GAMMA, "gam" },
	{ FEATURE_SHUTTER, "shut" },
	{ FEATURE_TEMPERATURE, "temp" },
	{ FEATURE_TRIGGER, "trig" },

	{ FEATURE_OPTICAL_FILTER, "opticalfilter" },
	GEN_UNDERSCORE (FEATURE_OPTICAL_FILTER, "opt", "filter"),
	GEN_UNDERSCORE (FEATURE_OPTICAL_FILTER, "optical", "filt"),
	GEN_UNDERSCORE (FEATURE_OPTICAL_FILTER, "opt", "filt"),

	GEN_UNDERSCORE (FEATURE_CAPTURE_SIZE, "capture", "size"),
	GEN_UNDERSCORE (FEATURE_CAPTURE_SIZE, "capt", "size"),
	GEN_UNDERSCORE (FEATURE_CAPTURE_SIZE, "cap", "size"),

	{ FEATURE_CAPTURE_QUALITY, "capturequality" },
	GEN_UNDERSCORE (FEATURE_CAPTURE_QUALITY, "capt", "quality"),
	GEN_UNDERSCORE (FEATURE_CAPTURE_QUALITY, "cap", "quality"),
	GEN_UNDERSCORE (FEATURE_CAPTURE_QUALITY, "capture", "qual"),
	GEN_UNDERSCORE (FEATURE_CAPTURE_QUALITY, "cap", "qual"),
	GEN_UNDERSCORE (FEATURE_CAPTURE_QUALITY, "capt", "qual")
};
int feature_alias_size = sizeof (feature_alias) / sizeof (struct lookup_struct);

/* get the feature number from a given name or alias */
int
feature_by_name (char *name)
{
	int i = 0;

	/* check *normal* names */
	for (i = 0; i < feature_table_size; i++)
	{
		if (strcmp (name, feature_table[i].name) == 0)
			return feature_table[i].value;
	}

	/* check aliases */
	for (i = 0; i < feature_alias_size; i++)
	{
		if (strcmp (name, feature_alias[i].name) == 0)
			return feature_alias[i].value;
	}

	/* no match */
	return 0;
}

/* get the feature name from the number */
char *
feature_by_num (int fnum)
{
	int i = 0;
	for (i = 0; i < feature_table_size; i++)
	{
		if (fnum == feature_table[i].value)
			return feature_table[i].name;
	}

	/* no match */
	return 0;
}

/* check if current camera supports this feature */
int
feature_available (int fnum)
{
	int value = 0;
	if (dc1394_is_feature_present (handle, current_node, FEATURE_CORRECT (fnum),
	    (dc1394bool_t *) &value) != DC1394_SUCCESS)
	{
		return 0; /* maybe the camera know nothing about it*/
	}
	return value;
}

/* get the value of the feature */
unsigned int
feature_get_value (int fnum)
{
	if (feature_available (fnum))
	{
		unsigned int value = 0;
		if (fnum == FEATURE_WHITE_BALANCE)
		{
			error_crit ("internal error: FEATURE_WHITE_BALANCE may not be used");
		}
		else if (fnum == FEATURE_WHITE_BALANCE_UB)
		{
			unsigned int dummy;
			if (dc1394_get_white_balance (handle, current_node, &value, &dummy) !=
			    DC1394_SUCCESS)
			{
				error_crit ("could not query feature although present");
			}
		}
		else if (fnum == FEATURE_WHITE_BALANCE_VR)
		{
			unsigned int dummy;
			if (dc1394_get_white_balance (handle, current_node, &dummy, &value) !=
			    DC1394_SUCCESS)
			{
				error_crit ("could not query feature although present");
			}
		}
		else
		{
			if (dc1394_get_feature_value (handle, current_node, fnum, &value) !=
			    DC1394_SUCCESS)
			{
				error_crit ("could not query feature although present");
			}
		}
		return value;
	}
	return 0;
}

/* get the minimal value of the feature */
unsigned int
feature_get_min (int fnum)
{
	if (feature_available (fnum))
	{
		unsigned int value = 0;
		if (dc1394_get_min_value (handle, current_node, FEATURE_CORRECT(fnum), &value) !=
		    DC1394_SUCCESS)
		{
			error_crit ("could not get minimal value");
		}
		return value;
	}
	return 0;
}

/* get the maximal value of the feature */
unsigned int
feature_get_max (int fnum)
{
	if (feature_available (fnum))
	{
		unsigned int value = 0;
		if (dc1394_get_max_value (handle, current_node, FEATURE_CORRECT(fnum), &value) !=
		    DC1394_SUCCESS)
		{
			error_crit ("could not get maximal value");
		}
		return value;
	}
	return 0;
}

/* check if the feature is in auto or manual mode */
int
feature_is_auto (int fnum)
{
	int value = 0;
	if (feature_available (fnum))
	{
		if (dc1394_is_feature_auto (handle, current_node, FEATURE_CORRECT (fnum),
		    (dc1394bool_t *)&value) != DC1394_SUCCESS)
		{
			error_crit ("could not check for auto mode");
		}
	}
	return value;
}

/* check if the feature supports auto mode */
int
feature_can_auto (int fnum)
{
	int value = 0;
	if (feature_available (fnum))
	{
		if (dc1394_has_auto_mode (handle, current_node, FEATURE_CORRECT (fnum),
		    (dc1394bool_t *)&value) != DC1394_SUCCESS)
		{
			error_crit ("could not check for auto mode");
		}
	}
	return value;
}

/* check if the feature supports manual mode */
int
feature_can_manual (int fnum)
{
	int value = 0;
	if (feature_available (fnum))
	{
		if (dc1394_has_manual_mode (handle, current_node, FEATURE_CORRECT (fnum),
		    (dc1394bool_t *)&value) != DC1394_SUCCESS)
		{
			error_crit ("could not check for manual mode");
		}
	}
	return value;
}

/* print some data about a feature (if full, then print more) */
void
print_feature (int fnum, int full)
{
	if (!feature_available (fnum))
	{
		printf ("# %s not available\n", feature_by_num (fnum));
		return;
	}

	if (full)
	{
		printf ("# full data for %s:\n", feature_by_num (fnum));
		printf ("# min: %u  max: %u\n", feature_get_min (fnum), feature_get_max (fnum));
		printf ("# available modes: %s%s\n", (feature_can_manual (fnum))?("MANUAL "):(""),
			(feature_can_auto (fnum))?("AUTO "):(""));
	}

	printf ("mode %s %s\n", feature_by_num (fnum), (feature_is_auto (fnum))?("auto"):("manual"));
	printf ("set %s %u\n", feature_by_num (fnum), feature_get_value (fnum));

	if (full) printf ("\n");
}

/* set the feature to auto mode */
void
feature_set_auto (int fnum)
{
	if (dc1394_auto_on_off (handle, current_node, FEATURE_CORRECT(fnum), 1) != DC1394_SUCCESS)
	{
		error_crit ("could not set auto mode");
	}
}

/* set the feature to manual mode */
void
feature_set_manual (int fnum)
{
	if (dc1394_auto_on_off (handle, current_node, FEATURE_CORRECT(fnum), 0) != DC1394_SUCCESS)
	{
		error_crit ("could not set manual mode");
	}
}

/* set the value of the feature */
void
feature_set_value (int fnum, unsigned int value)
{
	if (fnum == FEATURE_WHITE_BALANCE_UB)
	{
		if (dc1394_set_white_balance(handle, current_node, value,
		    feature_get_value (FEATURE_WHITE_BALANCE_VR)) != DC1394_SUCCESS)
		{
			error_crit ("could not change feature value");
		}
	}
	else if (fnum == FEATURE_WHITE_BALANCE_VR)
	{
		if (dc1394_set_white_balance(handle, current_node,
		    feature_get_value (FEATURE_WHITE_BALANCE_UB), value) != DC1394_SUCCESS)
		{
			error_crit ("could not change feature value");
		}
	}
	else
	{
		if (dc1394_set_feature_value (handle, current_node, FEATURE_CORRECT(fnum), value) !=
		    DC1394_SUCCESS)
		{
			error_crit ("could not change feature value");
		}
	}
}

/* get the protocol version supported by the camera */
char *
camera_get_version (void)
{
	quadlet_t value;
	static char version[20];

	if (dc1394_get_sw_version (handle, current_node, &value) != DC1394_SUCCESS)
	{
		error_crit ("could not read version");
	}

	switch (value)
	{
		case 0x000100: strcpy (version, "1.04"); break;
		case 0x000101: strcpy (version, "1.20"); break;
		case 0x000102: strcpy (version, "1.30"); break;
		default: strcpy (version, "unknown");
	}

	return version;
}

/* get the camera vendor */
char *
camera_get_vendor (void)
{
	static dc1394_camerainfo caminfo;
	if (dc1394_get_camera_info (handle, current_node, &caminfo) != DC1394_SUCCESS)
	{
		error_crit ("could not get camera info");
	}
	return caminfo.vendor;
}

/* get the camera model */
char *
camera_get_model (void)
{
	static dc1394_camerainfo caminfo;
	if (dc1394_get_camera_info (handle, current_node, &caminfo) != DC1394_SUCCESS)
	{
		error_crit ("could not get camera info");
	}
	return caminfo.model;
}



