/*
 * Copyright (C) 2002 Benjamin Hummel (benjamin@datamaze.de)
 *
 * camera.h - support functions for 1394 camera control
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifndef CAMERA_H
#define CAMERA_H

#include <libraw1394/raw1394.h>

/* own feature extensions */
#define FEATURE_WHITE_BALANCE_UB 5
#define FEATURE_WHITE_BALANCE_VR 6 /* these numbers are save, since features start at 416 */

/* and translation */
#define FEATURE_CORRECT(x) \
	((((x)==FEATURE_WHITE_BALANCE_UB)||((x)==FEATURE_WHITE_BALANCE_VR))?(FEATURE_WHITE_BALANCE):(x))

/* useful struct */
struct lookup_struct
{
	int value;
	char *name;
};

/* exported vars */
extern raw1394handle_t handle;
extern nodeid_t *camera_nodes;
extern int num_cameras;
extern int current_cam;
extern nodeid_t current_node;

/* export these tables */
extern int feature_table_size;
extern struct lookup_struct feature_table[];
extern int feature_alias_size;
extern struct lookup_struct feature_alias[];


/* get the feature number from a given name or alias */
int
feature_by_name (char *name);

/* get the feature name from the number */
char *
feature_by_num (int fnum);

/* check if current camera supports this feature */
int
feature_available (int fnum);

/* get the value of the feature */
unsigned int
feature_get_value (int fnum);

/* get the minimal value of the feature */
unsigned int
feature_get_min (int fnum);

/* get the maximal value of the feature */
unsigned int
feature_get_max (int fnum);

/* check if the feature is in auto or manual mode */
int
feature_is_auto (int fnum);

/* check if the feature supports auto mode */
int
feature_can_auto (int fnum);

/* check if the feature supports manual mode */
int
feature_can_manual (int fnum);

/* print some data about a feature (if full, then print more) */
void
print_feature (int fnum, int full);

/* set the feature to auto mode */
void
feature_set_auto (int fnum);

/* set the feature to manual mode */
void
feature_set_manual (int fnum);

/* set the value of the feature */
void
feature_set_value (int fnum, unsigned int value);

/* get the protocol version supported by the camera */
char *
camera_get_version (void);

/* get the camera vendor */
char *
camera_get_vendor (void);

/* get the camera model */
char *
camera_get_model (void);

#endif // CAMERA_H
