/*
 * Copyright (C) 2002 Benjamin Hummel (benjamin@datamaze.de)
 *
 * commands.c - implementation of internal commands
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <libdc1394/dc1394_control.h>

#include "main.h"
#include "error.h"
#include "commands.h"
#include "camera.h"

/* list of internal commands */
struct command_struct commands[] =
{
	{ "?",      cmd_help   },
	{ "help",   cmd_help   },
	{ "get",    cmd_get    },
	{ "quit",   cmd_quit   },
	{ "exit",   cmd_quit   },
	{ "mode",   cmd_mode   },
	{ "set",    cmd_set    },
	{ "camera", cmd_camera },
	{ "info",   cmd_info   }
};

int num_commands = sizeof (commands) / sizeof (struct command_struct);

/* some macros to unify help messages */
#define ISHELP(x) ((strcmp(x,"-h")==0) || (strcmp(x,"-?")==0) || (strcmp(x,"--help")==0))
#define HELP_BEGIN if((argc>1)&&(ISHELP(argv[1]))) {
#define HELP_GENERAL(x) printf ("%s: " x "\n", argv[0]);
#define HELP_USAGE(x)  printf ("usage: %s [options] " x "\n", argv[0]);
#define HELP_BEGIN_OPT {int i; printf ("options:\n");
#define HELP_MAX_OPT_LEN 15
#define HELP_OPTION(opt,exp) printf ("  " opt " "); \
	for(i=0;i<(HELP_MAX_OPT_LEN-strlen(opt));i++) printf(" "); printf(exp "\n");
#define HELP_OPTION_HELP HELP_OPTION("-h, --help", "display this help")
#define HELP_END_OPT printf ("\n"); }
#define HELP_TEXT(x) printf (x "\n");
#define HELP_END   return 0;}

/* internal commands */

int
cmd_help (int argc, char **argv, int line)
{
	int i = 0;
	char *help_param = "-h";
	char *help_argv[3] = { argv[1], help_param, 0 };

	HELP_BEGIN
	HELP_GENERAL ("display general help or help to a specific command")
	HELP_USAGE("[command]")
	HELP_BEGIN_OPT
	HELP_OPTION_HELP
	HELP_END_OPT
	HELP_TEXT ("command is either \"features\", \"alias\" or the name of a command")
	HELP_TEXT ("To get a list of all commands use \"help\" without parameters.")
	HELP_END

	if (argc == 1)
	{
		printf ("Available commands are:\n");
		printf ("  camera  get  help  info  mode  quit  set\n\n");
		printf ("To get further help try \"help <command>\"\n");
		return 0;
	}

	for (i = 0; i < num_commands; i++)
	{
		if (strcmp (commands[i].name, argv[1]) == 0)
		{
			return ((commands[i].fct)(2, help_argv, line));
		}
	}

	if (strcmp (argv[1], "features") == 0)
	{
		printf ("known camera features:\n");
		for (i = 0; i < feature_table_size; i++)
			printf ("  %s\n", feature_table[i].name);
		printf ("\nFor a list of available aliases type \"help alias\"\n");
		return 0;
	}
	else if (strcmp (argv[1], "alias") == 0)
	{
		int j;
		printf ("these are tha available aliases:\n");
		for (i = 0; i < feature_table_size; i++)
		{
			printf ("  %s:\n", feature_table[i].name);
			for (j = 0; j < feature_alias_size; j++)
			{
				if (feature_alias[j].value == feature_table[i].value)
				{
					printf ("    %s\n", feature_alias[j].name);
				}
			}
		}
		printf ("\nFor a list of featues only type \"help features\"\n");
		return 0;
	}

	/* no match ==> error */
	return error_parse (line, "no help available for unknown command", argv[1]);
}

int
cmd_get (int argc, char **argv, int line)
{
	int i = 0;
	int long_list = 0;

	HELP_BEGIN
	HELP_GENERAL ("display the settings of a feature")
	HELP_USAGE("feature [feature[..]]")
	HELP_BEGIN_OPT
	HELP_OPTION_HELP
	HELP_OPTION ("-l, --long", "display longer information")
	HELP_END_OPT
	HELP_TEXT ("feature is either \"all\", \"available\" or the name of a feature")
	HELP_TEXT ("To get a list of all features use \"help features\" or \"help alias\".")
	HELP_END

	if (argc < 2) return error_parse (line, "get", "no parameters");

	for (i = 1; i < argc; i++)
	{
		int feature = feature_by_name (argv[i]);
		if (feature != 0)
		{
				print_feature (feature, long_list);
		}
		else if ((strcmp (argv[i], "avail") == 0) || (strcmp (argv[i], "available") == 0))
		{
			int j = 0;
			for (j = 0; j < feature_table_size; j++)
			{
				if (feature_available (feature_table[j].value))
					print_feature (feature_table[j].value, long_list);
			}
		}
		else if (strcmp (argv[i], "all") == 0)
		{
			int j = 0;
			for (j = 0; j < feature_table_size; j++)
			{
				print_feature (feature_table[j].value, long_list);
			}
		}
		else if ((strcmp (argv[i], "-l") == 0) || (strcmp (argv[i], "--long") == 0))
		{
			long_list = 1;
		}
		else
		{
			return error_parse (line, "get: unknown feature", argv[i]);
		}
	}

	return 0;
}

int
cmd_quit (int argc, char **argv, int line)
{
	HELP_BEGIN
	HELP_GENERAL ("exit the program")
	HELP_USAGE("")
	HELP_BEGIN_OPT
	HELP_OPTION_HELP
	HELP_END_OPT
	HELP_END

	save_exit (0);
	return 0;
}


int
cmd_mode (int argc, char **argv, int line)
{
	int feature = 0;
	int auto_mode = 0;
	int force = 0;
	int i = 1;

	HELP_BEGIN
	HELP_GENERAL ("set the mode of a given feature")
	HELP_USAGE("feature mode")
	HELP_BEGIN_OPT
	HELP_OPTION ("-f, --force", "even set the mode if not supported")
	HELP_OPTION_HELP
	HELP_END_OPT
	HELP_TEXT ("feature is either \"all\", \"available\" or the name of a feature")
	HELP_TEXT ("To get a list of all features use \"help features\" or \"help alias.\"")
	HELP_TEXT ("mode should be either \"manual\" or \"auto\".")
	HELP_TEXT ("The abbreviations \"m\" and \"a\" are also supported.")
	HELP_END

	while ((strcmp (argv[i], "-f")==0) || (strcmp (argv[i], "--force")==0))
	{
		force = 1;
		i++;
	}
	if (argc < (i+2)) return error_parse (line, argv[0], "not enough parameters");

	feature = feature_by_name (argv[i]);
	if (feature == 0) return error_parse (line, "invalid feature", argv[i]);

	if ((strcmp (argv[i+1], "a") == 0) || (strcmp (argv[i+1], "auto") == 0))
	{
		auto_mode = 1;
	}
	else if ((strcmp (argv[i+1], "m") == 0) ||
	         (strcmp (argv[i+1], "man") == 0) || (strcmp (argv[i+1], "manual") == 0))
	{
		auto_mode = 0;
	}
	else
	{
		return error_parse (line, "invalid mode", argv[i+1]);
	}

	if (!force)
	{
		if ((auto_mode && !feature_can_auto (feature)) ||
		    (!auto_mode && !feature_can_manual (feature)))
		{
			printf (APP_NAME ":%s: %s does not support %s mode, ignoring\n", argv[0],
				feature_by_num (feature), (auto_mode)?("auto"):("manual"));
			return 0;
		}
	}

	if (auto_mode) feature_set_auto (feature);
	else feature_set_manual (feature);

	return 0;
}

int
cmd_set (int argc, char **argv, int line)
{
	int feature = 0;
	unsigned int value = 0;
	int force = 0;
	int i = 1;

	HELP_BEGIN
	HELP_GENERAL ("change the value of a camera feature")
	HELP_USAGE("feature value")
	HELP_BEGIN_OPT
	HELP_OPTION ("-f, --force", "even set value if too small or too big")
	HELP_OPTION_HELP
	HELP_END_OPT
	HELP_TEXT ("feature is either \"all\", \"available\" or the name of a feature")
	HELP_TEXT ("To get a list of all features use \"help features\" or \"help alias.\"")
	HELP_TEXT ("value should be a positive number")
	HELP_END

	while ((strcmp (argv[i], "-f")==0) || (strcmp (argv[i], "--force")==0))
	{
		force = 1;
		i++;
	}
	if (argc < (i+2)) return error_parse (line, argv[0], "not enough parameters");

	feature = feature_by_name (argv[i]);
	if (feature == 0) return error_parse (line, "invalid feature", argv[i]);

	if (!isdigit(argv[i+1][0])) return error_parse (line, "invalid value", argv[i+1]);
	value = atoi (argv[i+1]);
	if ((value == 0) && (argv[i+1][0] != '0'))
		return error_parse (line, "invalid value", argv[i+1]);

	if (!force)
	{
		if (value < feature_get_min (feature))
		{
			printf ("value was too small, corrected\n");
			value = feature_get_min (feature);
		}
		if (value > feature_get_max (feature))
		{
			printf ("value was too big, corrected\n");
			value = feature_get_max (feature);
		}
	}

	feature_set_value (feature, value);

	return 0;
}

int
cmd_camera (int argc, char **argv, int line)
{
	HELP_BEGIN
	HELP_GENERAL ("change the current camera")
	HELP_USAGE("camera")
	HELP_BEGIN_OPT
	HELP_OPTION_HELP
	HELP_END_OPT
	HELP_TEXT ("camera should be a number between 0 and max-1, where max is the number of cameras available.")
	HELP_TEXT ("To get max use \"camera max\".");
	HELP_END

	if (argc < 2) return error_parse (line, "camera", "no parameters");

	if (strcmp (argv[1], "max") == 0)
	{
		printf ("%d\n", num_cameras);
	}
	else
	{
		int num = atoi (argv[1]);
		if ((num == 0) && (argv[1][0] != '0'))
		{
			return error_parse (line, "invalid camera", argv[1]);
		}
		if (num >= num_cameras)
		{
			error_crit ("invalid camera selected");
		}
		current_cam = num;
		current_node = camera_nodes[current_cam];
	}

	return 0;
}

int
cmd_info (int argc, char **argv, int line)
{
	int i = 0;

	HELP_BEGIN
	HELP_GENERAL ("display several informations about the camera")
	HELP_USAGE("info [info[..]]")
	HELP_BEGIN_OPT
	HELP_OPTION_HELP
	HELP_END_OPT
	HELP_TEXT ("info should be one of \"vendor\", \"model\", \"version\" or \"all\"")
	HELP_END

	if (argc < 2)
	{
		return error_parse (line, "info", "no parameters");
	}

	for (i = 1; i < argc; i++)
	{
		if (strcmp (argv[i], "version") == 0)
		{
			printf ("# version: %s\n", camera_get_version());
		}
		else if (strcmp (argv[i], "vendor") == 0)
		{
			printf ("# vendor:  %s\n", camera_get_vendor());
		}
		else if (strcmp (argv[i], "model") == 0)
		{
			printf ("# model:   %s\n", camera_get_model());
		}
		else if (strcmp (argv[i], "all") == 0)
		{
			char *all_commands[] =
				{ "vendor", "model", "version" };
			int all_count = sizeof (all_commands) / sizeof (char *);
			int j;
			char *local_argv[3] = {"info", 0, 0};

			for (j = 0; j < all_count; j++)
			{
				local_argv[1] = all_commands[j];
				cmd_info (2, local_argv, line);
			}
		}
		else
		{
			return error_parse ( line, "unknown info", argv[i] );
		}
	}

	return 0;
}




