/*
 * Copyright (C) 2002 Benjamin Hummel (benjamin@datamaze.de)
 *
 * main.c - the main file
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include <stdio.h>
#include <signal.h>
#include <unistd.h>

#include <libdc1394/dc1394_control.h>
#include <libraw1394/raw1394.h>

#include "main.h"
#include "error.h"
#include "commands.h"
#include "camera.h"

/* remove comments (everything behind # or / or ;) and CR or LF */
void
uncomment (char *line)
{
	char *c = line;
	while (*c != 0) {
		if ((*c == '#') || (*c == '/') || (*c == ';')
		    || (*c == '\n') || (*c == '\r'))
		{
			*c = 0;
			return;
		}
		c++;
	}
}

/* replace tabs with normal spaces */
void
untab (char *line)
{
	char *c = line;
	while (*c != 0)
	{
		if (*c == '\t') *c = ' ';
		c++;
	}
}

/* parse one line */
int
process_line (char *line, int line_num)
{
	char *cmd = line;
	char *argv[MAX_LINE_LENGTH/2+3];
	int argc = 0;
	int i = 0;

	uncomment (cmd);
	untab (cmd);
	while (*cmd == ' ') cmd++;
	if (cmd[0] == 0) return 0; /* ignore empty lines */

	for (;;)
	{
		argv[argc++] = cmd;
		while ((*cmd != ' ') && (*cmd != 0)) cmd++;
		if (*cmd == ' ')
		{
			while (*cmd == ' ') *cmd++ = 0;
		}
		if (*cmd == 0)
		{
			argv[argc] = 0;
			break;
		}
	}

	for (i = 0; i < num_commands; i++)
	{
		if (strcmp (commands[i].name, argv[0]) == 0)
		{
			return ((commands[i].fct) (argc, argv, line_num));
		}
	}

	/* if nothing matched ==> error */
	return error_parse (line_num, "unknown command", argv[0]);
}

/* print the usual usage stuff */
void
print_help (void)
{
	printf (APP_NAME " " APP_VERSION ": control some parameters of a IEEE1394 digital camera\n");
	printf ("Usage: " APP_NAME " [options]\n");
	printf ("  -d, --direct     read commands as command line arguments\n");
	printf ("  -h, --help       this help\n");
	printf ("  -n, --noread     don't enter interactive mode (useful with -d)\n");
	printf ("  -s, --strict     exit, if a single command reported an error\n");
	printf ("\nIf called without -d you enter interactive mode. \n");
	printf ("Type help then for further information\n");
}

/* catch SIGINT to allow for clean exit */
void
my_sigfunc (int signal)
{
	printf (APP_NAME ": signal detected.\n");
	save_exit (0);
}

/* the prgrams main function */
int
main (int argc, char **argv)
{
	char line[MAX_LINE_LENGTH+2];
	int line_num = 1;
	FILE *inf = stdin;
	int is_interactive = 0;
	int arg_commands = 0;
	int i = 0;
	int read_stdin = 1;

	/* trap signals */
	if (signal (SIGINT, my_sigfunc) == SIG_ERR) error_crit ("could not setup signal handler");

	/* check if in interactive mode (0 corresponds to stdin) */
	if (isatty(0)) is_interactive = 1;
	else is_interactive = 0;

	for (i = 1; i < argc; i++)
	{
		if ((strcmp (argv[i], "-h") == 0) || (strcmp (argv[i], "--help") == 0))
		{
			print_help();
			return 0;
		}
		else if ((strcmp (argv[i], "-s") == 0) || (strcmp (argv[i], "--strict") == 0))
		{
			fail_on_error = 1;
		}
		else if ((strcmp (argv[i], "-d") == 0) || (strcmp (argv[i], "--direct") == 0))
		{
			arg_commands = i+1;
			break;
		}
		else if ((strcmp (argv[i], "-n") == 0) || (strcmp (argv[i], "--noread") == 0))
		{
			read_stdin = 0;
		}
		else
		{
			printf (APP_NAME ": unknown parameter: %s\n", argv[i]);
			return APP_ERROR;
		}
	}

	handle = dc1394_create_handle(0);
	if (handle == 0) error_crit ("could not create raw1394 handle");

	camera_nodes = dc1394_get_camera_nodes(handle,&num_cameras,0);
	if (num_cameras < 1) error_crit ("no cameras found");
	current_node = camera_nodes[0];

	if (arg_commands > 0)
	{
		for (i = arg_commands; i < argc; i++)
		{
			process_line (argv[i], -(argc-i));
		}
	}

	if (read_stdin)
	{
		if (is_interactive) printf (PROMPT);
		while (fgets(line, MAX_LINE_LENGTH, inf) != 0)
		{
			process_line (line, line_num++);
			if (is_interactive) printf (PROMPT);
		}
		if (is_interactive) printf ("\n");
	}

	if (handle != 0) raw1394_destroy_handle (handle);
	return 0;
}
